#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai


__license__   = 'GPL v3'
__copyright__ = '2025, Digital Assassins'
__docformat__ = 'restructuredtext en'

if False:
    # This is here to keep my python error checker from complaining about
    # the builtin functions that will be defined by the plugin loading system
    # You do not need this code in your plugins
    get_icons = get_resources = None

# The class that all interface action plugins must inherit from
from calibre.gui2.actions import InterfaceAction
from PyQt5.QtCore import QTimer

from operator import attrgetter
from calibre.ebooks.metadata.sources.prefs import msprefs        
from calibre.customize.ui import all_metadata_plugins, default_disabled_plugins, disable_plugin, enable_plugin, is_disabled
from calibre.gui2 import error_dialog

class DownloadAIMetadata(InterfaceAction):

    name = 'AI Metadata'
    action_spec = (name, None,
            'Get Metadata from the Book Content using the AI Metadata Plugin', None)
    action_type = 'current'

    def genesis(self):
        base_plugin_object = self.interface_action_base_plugin
        self.edit_metadata = self.gui.iactions['Edit Metadata']
        self.icon = get_icons('images/download-metadata.png', 'Download AI Metadata')
        self.qaction.setIcon(self.icon)
        self.qaction.triggered.connect(self.start_metadata_action)
    
    def check_ai_metadata_plugin_exists(self):
        for plugin in self.plugins:
            if plugin.name == "AI Metadata":
                self.is_installed = True
                return
        self.is_installed = False
    
    def get_metadata_plugins(self):
        print("Getting Metadata Plugins: ")
        ## run through all metadata plugins
        self.plugins = list(all_metadata_plugins())        
        self.plugins.sort(key=attrgetter('name'))
        self.enabled_plugins = []
        for plugin in self.plugins:
            print(plugin)
            if is_disabled(plugin.name) == False:
                print("Enabled: " + plugin.name)
                self.enabled_plugins.append(plugin.name)
    
    def disable_all_plugins(self):        
        for plugin in self.enabled_plugins:
            print("Disabling: " + plugin)
            disable_plugin(plugin)
        print("Enabling AI Metadata Plugin")
        enable_plugin("AI Metadata")
        #update_sources()
        return
    
    def enable_all_plugins(self):
        for plugin in self.enabled_plugins:
            print("Enabling: " + plugin)
            enable_plugin(plugin)
        print("Disabling AI Metadata Plugin")
        disable_plugin("AI Metadata")
        #update_sources()
        return
    
    def get_selected_books_in_main_gui(self):
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, _('Cannot download metadata'),
                        _('No books selected'), show=True)
        db = self.gui.library_view.model().db
        ids = [db.id(row.row()) for row in rows]
        return ids
        
        
    def start_metadata_action(self):
        
        print("-----------------------------------")
        self.get_metadata_plugins()
        print("-----------------------------------")
        self.check_ai_metadata_plugin_exists()
        print("-----------------------------------")
        
        if self.is_installed == False:
            return error_dialog(self.gui, _('AI Metadata Plugin'),
                        _('AI Metadata Plugin is not installed'), show=True)
                        
        # disable the other plugins
        self.disable_all_plugins()
        #QTimer.singleShot(500, None);
        # get the ids
        ids = self.get_selected_books_in_main_gui()
        print(ids)
        self.start_download(ids)
        # re enable the other plugins
        #QTimer.singleShot(10000, None);
        return
    
    def start_download(self, ids, ensure_fields=None):        
        #self.edit_metadata.download_metadata(ids)
        from calibre.gui2.metadata.bulk_download import Job, download
        from calibre.ptempfile import PersistentTemporaryDirectory, PersistentTemporaryFile
        from calibre.utils.localization import ngettext
        from calibre.gui2 import Dispatcher
        
        tf = PersistentTemporaryFile('_metadata_bulk.log')
        tf.close()
        
        job = Job('metadata bulk download',
            ngettext(
                'Download metadata for one book',
                'Download metadata for {} books', len(ids)).format(len(ids)),
            download, (ids, tf.name, self.gui.current_db, True, False,
                ensure_fields), {}, Dispatcher(self.metadata_downloaded) )
        job.metadata_and_covers = (True, False)
        job.download_debug_log = tf.name
        self.gui.job_manager.run_threaded_job(job)
        self.gui.status_bar.show_message(_('Metadata download started'), 3000)
    
    def metadata_downloaded(self, job):
        self.edit_metadata.metadata_downloaded(job)
        self.enable_all_plugins()
    
    def show_dialog(self):
        return

    def apply_settings(self):
        return